from PyQt5.QtCore import QSize, Qt, QThread
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QGridLayout, QPushButton, QSizePolicy, QSlider, QSpacerItem, QToolBar, QWidget

import instance
from jumpslider import JumpSlider
from lib import constants
from lib.constants import ResPath
from lib.i18n import lu
from seismogramwidget import SeismogramWidget
from util import renderutil
from widgets.materialeditor.emitterplacement import EmitterPlacement
from widgets.materialeditor.receiverplacement import ReceiverPlacement
from widgets.materialeditor.settingcontainer import ImageWidget, SettingContainer, WaveOverlayWidget


# TODO: only stop looping for the first time
class AnimationWidget(QWidget):
    cycles = 2

    def __init__(self, parent=None):
        QWidget.__init__(self, parent)

        self._timerId = None
        self._cycleCounter = 0
        self._frames = []

        # app layout
        gridLayout = QGridLayout(self)
        gridLayout.setVerticalSpacing(0)  # we don't want vertical space between the widgets
        
        self._emitterPlacement = EmitterPlacement()
        self._emitterPlacement.setDraggable(False)
        self._receiverPlacement = ReceiverPlacement()
        self._receiverPlacement.setDraggable(False)
        # create the aspect ratio container
        self.settingContainer = SettingContainer(self._emitterPlacement, self._receiverPlacement, ImageWidget(), instance.getMaterialAspectRatio())
        self.settingContainer.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Expanding)
        gridLayout.addWidget(self.settingContainer, 0, 1)
        
        self.waveWidget = WaveOverlayWidget(self)
    
        self.initToolbar(gridLayout)   

        # spacing between canvas and seismogram
        gridLayout.addItem(QSpacerItem(0, 10, vPolicy=QSizePolicy.Fixed), 1, 0, 1, 2)

        # slider
        self.slider = JumpSlider(Qt.Horizontal, tickPosition=QSlider.TicksBelow, valueChanged=self.sliderChanged)
        self.slider.setTickInterval(1)
        self.slider.setPageStep(1)
        self.slider.setInvertedControls(True)
        gridLayout.addWidget(self.slider, 3, 1)

        # seismogram widget
        self._seismogram = SeismogramWidget(self.slider, isForward=True)
        gridLayout.addWidget(self._seismogram, 2, 1)

        # space below seismogram
        gridLayout.addItem(QSpacerItem(0, 8), 4, 1)
        # start inversion button
        self.bStartInversion = QPushButton(lu("buttonStartInversion"), clicked=self.gotoInversion)
        gridLayout.addWidget(self.bStartInversion, 5, 1)

    def initToolbar(self, gridLayout):
        # toolbar
        toolbar = QToolBar(orientation=Qt.Vertical)
        toolbar.setIconSize(QSize(32, 32))

        # loop action
        self._aLoop = QAction(QIcon(ResPath.icons + "loop.png"), lu("actionLoop"), self, checkable=True, checked=True)
        toolbar.addAction(self._aLoop)
        # stop action
        toolbar.addAction(QAction(QIcon(ResPath.icons + "stop.png"), lu("actionStop"), self, triggered=self.stop))
        # play/pause action
        icon = QIcon(ResPath.icons + "play.png")
        icon.addFile(ResPath.icons + "pause.png", state=QIcon.On)
        trigger = lambda: self.start() if self._aPlayPause.isChecked() else self.pause()  # cant use self.toggle because action state is changed before call
        self._aPlayPause = QAction(icon, lu("actionPlay"), self, checkable=True, shortcut=" ", triggered=trigger)
        toolbar.addAction(self._aPlayPause)
        gridLayout.addWidget(toolbar, 2, 0, 2, 1, alignment=Qt.AlignBottom)

    def resizeEvent(self, event):
        QWidget.resizeEvent(self, event)
        waveRect = self.settingContainer.getRectForWaves()
        self.waveWidget.setGeometry(waveRect)

    def timerEvent(self, event):
        k = self.slider.value()
        nextValue = (k + 1) % (constants.FRAME_COUNT)
        
        # next frame not available jet
        if len(self._frames) <= nextValue:
            return

        # next cycle
        if nextValue < k:
            self._cycleCounter += 1
            if not self._aLoop.isChecked() or self._cycleCounter >= AnimationWidget.cycles:
                self.pause()
        # enough animation cycles
        if self._cycleCounter >= AnimationWidget.cycles:
            #k = (self.slider.maximum() - self.slider.minimum()) // 3
            k = 0
        else:
            k = nextValue

        self.slider.setValue(k)

#         self.waveWidget.setFrame(self._frames[k][0]) # gets called by slider.valueChanged
#         self.update() # gets called by slider.valueChanged

    def sliderChanged(self):
        k = self.slider.value()
        if len(self._frames) == constants.FRAME_COUNT: # solve finished -> don't update seismogram
            pass
        elif k < len(self._frames):
            self._seismogram.set_current_seismogram(self._frames[-1][1])
        else: # frame not available jet
            return
        
        self.waveWidget.setFrame(self._frames[k][0])
        self.update()

    def toggle(self):
        if not self._aPlayPause.isChecked():
            self.start()
        else:
            self.pause()

    def start(self):
        if not self._timerId:
            self._cycleCounter = 0
            self._timerId = self.startTimer(50)
            self._aPlayPause.setChecked(True)
            self._aPlayPause.setText(lu("actionPause"))

    def pause(self):
        if self._timerId:
            self.killTimer(self._timerId)
            self._timerId = None
            self._aPlayPause.setChecked(False)
            self._aPlayPause.setText(lu("actionPlay"))

    def stop(self):
        self._cycleCounter = 0
        self.pause()
        self.slider.setValue(0)

    def loadFromData(self, material, emitterPos, receiverPos, centerXEmitRec):
        self.reset()
        self.settingContainer.setCenterXOfEmitRec(centerXEmitRec)
        
        self._realMaterial = material
        # set the background (scene) Image of this animation widget
        self.settingContainer.setMaterialImage(renderutil.arrayToQImage(material))

        # start forward solve
        self._frames = instance.computationManager().solveForward(material, emitterPos, receiverPos)
        PollingThread(self, self._frames, self.bStartInversion).start()

    def wheelEvent(self, event):
        self.slider.wheelEvent(event)

    def reset(self):
        self.stop()
        self._aLoop.setChecked(True)
        self.bStartInversion.setEnabled(False)

    def gotoInversion(self):
        self.pause()
        self.bStartInversion.setEnabled(False)       
        centerXEmitRec = self.settingContainer.getCenterXEmitRec()
        # TODO: 
        # Here we have to wait for the last seismogram, possible solutions:
        # 1) Wait in a Thread: tried, but failed (second window opened wtf?)
        # 2) Enable InversionButton if all seismograms are rendered. -> use PollingThread
        
        seismoData = self._frames[-1][2]
        
        # show 'loading'-overlay
#         overlay = instance.window().getOverlay()
#         overlay.addText((self.width() / 2, self.height() / 2), lu("overlayLoading"))
#         overlay.show(animate=False)
#         QCoreApplication.instance().processEvents()

        instance.window().switchToInversion(self._realMaterial, seismoData, centerXEmitRec)

        # hide overlay when calculation is done
#         overlay.hide(animate=False)


class PollingThread(QThread):
    def __init__(self, parent, frames, button):
        QThread.__init__(self, parent=parent)
        self.finished.connect(self.deleteLater)
        
        self._frames = frames
        self._button = button

    def run(self):
        # TODO: busy waiting is bad!?
        while len(self._frames) < constants.FRAME_COUNT:
            self.msleep(500)
        
        # enable 'start inversion'-button
        self._button.setEnabled(True)
        print "finished polling"        
