from PyQt5.QtCore import QRectF, Qt
from PyQt5.QtGui import QPainterPath, QRadialGradient, QColor, QPen
from PyQt5.QtWidgets import QGraphicsItem, QStyle
from util import utils


class NodeItem(QGraphicsItem):
    def __init__(self, graph, radius=10, stationary=False):
        QGraphicsItem.__init__(self)
        
        if stationary:
            self.setVisible(False)
        else:
            self.setAcceptedMouseButtons(Qt.LeftButton)
            self.setFlag(QGraphicsItem.ItemIsMovable)
            self.setFlag(QGraphicsItem.ItemSendsGeometryChanges)
            self.setCacheMode(QGraphicsItem.DeviceCoordinateCache)

        # ensure that the nodes are always stacked on top
        self.setZValue(1)
        
        self._graph = graph
        
        self._radius = radius
        self._rect = QRectF(-self._radius, -self._radius, 2*self._radius, 2*self._radius)
        
        self._leftNode = None
        self._rightNode = None

    def radius(self):
        return self._radius

    def setLeft(self, node):
        self._leftNode = node
        node._rightNode = self

    def setRight(self, node):
        self._rightNode = node
        node._leftNode = self

    def getLeft(self):
        return self._leftNode

    def getRight(self):
        return self._rightNode

    def boundingRect(self):
        return self._rect

    def shape(self):
        path = QPainterPath()
        path.addEllipse(self._rect)
        return path

    def paint(self, painter, option, widget):
        gradient = QRadialGradient(-3, -3, 10)
        colors = (QColor(Qt.green), QColor(Qt.darkGreen))
        # color when dragged
        if option.state & QStyle.State_Sunken:
            gradient.setCenter(4, 4)
            gradient.setFocalPoint(2, 2)
            gradient.setColorAt(1, colors[0].lighter(120))
            gradient.setColorAt(0, colors[1].lighter(120))
        # color when not dragged
        else:
            gradient.setColorAt(0, colors[0])
            gradient.setColorAt(1, colors[1])
   
        painter.setBrush(gradient)
        painter.setPen(QPen(Qt.black, 0))
        painter.drawEllipse(self._rect)

    def itemChange(self, change, value):
        if change == QGraphicsItem.ItemPositionHasChanged:
            x = self.pos().x()
            y = self.pos().y()
            
            if self.scene():
                sceneRect = self.scene().sceneRect()
                x = utils.clamp(x, sceneRect.left(), sceneRect.right())
                y = utils.clamp(y, sceneRect.top(), sceneRect.bottom())
            
            # keep this node in between it's two neighbours
            if self._leftNode:
                x = max(x, self._leftNode.x() + 5)
            if self._rightNode:
                x = min(x, self._rightNode.x() - 5)
            
            self.setPos(x, y)
            self._graph.recalculate()
    
        return QGraphicsItem.itemChange(self, change, value)

    def mousePressEvent(self, event):
        self.update()
        QGraphicsItem.mousePressEvent(self, event)

    def mouseReleaseEvent(self, event):
        self.update()
        QGraphicsItem.mouseReleaseEvent(self, event)
