from PyQt5.Qt import Qt
from PyQt5.QtCore import QAbstractAnimation, QPoint, QPropertyAnimation, QPointF
from PyQt5.QtGui import QColor, QFont, QFontMetrics, QPainter, QPalette
from PyQt5.QtWidgets import QGraphicsOpacityEffect, QWidget

from util import utils
from lib import constants


class OverlayWidget(QWidget):
    """
    represents an overlay widget to display images- or text-overlays. It should be created on top of all other widgets.
    """
    def __init__(self, parent):
        """
        @param parent: the parent of this widget. An overlay without a parent makes no sense
        """
        QWidget.__init__(self, parent)
        
        palette = QPalette(self.palette())
        palette.setColor(palette.Background, Qt.transparent)
        self.setPalette(palette)
        
        # ratios to display images, text, ... at the right size/pos in regard to the current window size
        self._ratioW = 1.0
        self._ratioH = 1.0
        
        self._imageQueue = []
        self._textQueue = {}
        self._layoutedTextQueue = {} # {20: [(QRect(100,100,400,400), "Hallo\nWelt", Qt.AlignCenter)]}
        
        opacityEffect = QGraphicsOpacityEffect(self)
        self.setGraphicsEffect(opacityEffect)
        # animation for fade-in
        self._animFadeIn = QPropertyAnimation(opacityEffect, "opacity")
        self._animFadeIn.setDuration(500)
        self._animFadeIn.setStartValue(0.0)
        self._animFadeIn.setEndValue(1.0)
        # animation for fade-out
        self._animFadeOut = QPropertyAnimation(opacityEffect, "opacity", finished=self._hide)
        self._animFadeOut.setDuration(500)
        self._animFadeOut.setStartValue(1.0)
        self._animFadeOut.setEndValue(0.0)

        self._baseOpacity = 150

    def paintEvent(self, event):
        painter = QPainter(self)
        painter.setRenderHint(QPainter.Antialiasing)
        painter.fillRect(event.rect(), QColor(255, 255, 255, self._baseOpacity))
        
        for pos, image in self._imageQueue:
            p = QPoint(pos.x() * self._ratioW, pos.y() * self._ratioH)
            i = image.scaled(image.width() * self._ratioW, image.height() * self._ratioH)
            painter.drawImage(p, i)

        for textSize, textList in self._textQueue.items():
            font = QFont()
            font.setPixelSize(textSize)
            painter.setFont(font)
            fontHeight = painter.fontMetrics().height()
        
            for pos, text, color in textList:
                painter.setPen(QColor(*color))
                y = pos.y() * self._ratioH
                for line in text.split("\\n"):
                    painter.drawText(pos.x() * self._ratioW, y, line)
                    y += fontHeight
        
        color = (0, 0, 0)
        for textSize, textList in self._layoutedTextQueue.items():
            font = QFont()
            font.setPixelSize(textSize)
            painter.setFont(font)
            fontHeight = painter.fontMetrics().height()
        
            for rect, text, color, flags in textList:
                painter.setPen(QColor(*color))
                painter.drawText(rect, flags, text)

    # TODO: maybe add original images AND size (instead of only images) to avoid blurryness when scaling
    # TODO: scale text around it's midpoint rather then it's upper left corner
    def resizeEvent(self, event):
        QWidget.resizeEvent(self, event)
        if not event.oldSize():
            return

        # update ratios
        self._ratioW = self.width() / float(constants.DEFAULT_WIDTH)
        self._ratioH = self.height() / float(constants.DEFAULT_HEIGHT)
            
    def addImages(self, positions, images):
        """
        add images at their specified positions to the image queue. They will be displayed on the next show() call.
        @param positions: an iterable of tuples of (x, y)-coordinates or a single (x, y)-tuple specifying the positions of each image
        @param images: an iterable of QImages or a single image that will be displayed
        """
        # when only a single tuple is given
        if not utils.isIterable(positions[0]):
            positions = (positions,)
        # only a single text was provided
        if not utils.isIterable(images):
            images = (images,)
        if len(positions) != len(images):
            print "Number of positions does not match the number of images. {} position, {} texts".format(len(positions), len(images)) 
            return
        # create QPoints from position tuples
        points = [QPoint(*pos) for pos in positions]
        self._imageQueue.extend(zip(points, images))

    def addText(self, positions, texts, textSize=30, color=(0, 0, 0)):
        """
        add texts at their specified positions to the text queue. They will be displayed on the next show() call.
        @param positions: an iterable of tuples of (x, y)-coordinates or a single (x, y)-tuple specifying the positions of each text
        @param texts: an iterable of Strings or a single String that will be displayed
        @keyword textSize: (optional) the size of the text
        @keyword color: (optional) the text color
        """
        # when only a single tuple is given
        if not utils.isIterable(positions[0]):
            positions = (positions,)
        # only a single text was provided (cant check for iterable because strings are iterable)
        if isinstance(texts, basestring): # Python 3: isinstance(arg, str)
            texts = (texts,)
        if len(positions) != len(texts):
            print "Number of positions does not match the number of texts. {} position, {} texts".format(len(positions), len(texts))
            return
        # create QPoints from position tuples
        points = [QPoint(*pos) for pos in positions]
        # check if key is not present yet
        if textSize not in self._textQueue:
            self._textQueue[textSize] = []
        self._textQueue[textSize].extend(zip(points, texts, [color]*len(texts)))

    def addLayoutedText(self, rects, texts, textSize=30, color=(0, 0, 0), flags=0):
        """
        add texts at their specified rects to the text queue. They will be displayed on the next show() call.
        @param rects: an iterable of QRects or a single QRect specifying the geometry of each text
        @param texts: an iterable of Strings or a single String that will be displayed
        @keyword textSize: (optional) the size of the text
        @keyword color: (optional) the text color
        @keyword flags: (optional) flags for the QPainter.drawText
        """
        # when only a single tuple is given
        if not utils.isIterable(rects):
            rects = (rects,)
        # only a single text was provided (cant check for iterable because strings are iterable)
        if isinstance(texts, basestring): # Python 3: isinstance(arg, str)
            texts = (texts,)
        if len(rects) != len(texts):
            print "Number of rects does not match the number of texts. {} rects, {} texts".format(len(rects), len(texts))
            return
        # check if key is not present yet
        if textSize not in self._layoutedTextQueue:
            self._layoutedTextQueue[textSize] = []
        self._layoutedTextQueue[textSize].extend(zip(rects, texts, [color]*len(texts), [flags]*len(texts)))

    def getStringWidth(self, text, fontSize=30):
        """
        @param text: the string which width should be calculated
        @keyword fontSize: the fontSize used to calculate the width, default=30.
        @return: the width of the text when using a font of given size
        """
        font = QFont()
        font.setPixelSize(fontSize)
        fInfo = QFontMetrics(font)
        return fInfo.width(text)

    def mousePressEvent(self, event):
        # fade out on click
        if event.button() == Qt.LeftButton:
            # prevent closing before fade-in has finished
            fadeInFlag = self._animFadeIn.state() == QAbstractAnimation.Stopped
            fadeOutFlag = self._animFadeOut.state() == QAbstractAnimation.Stopped
            if fadeInFlag and fadeOutFlag:
                self._animFadeOut.start()

    def show(self, animate=True, opacity=150):
        if self.isVisible():
            return

        self._baseOpacity = opacity

        if animate:
            self._animFadeIn.start()
        else:
            self.graphicsEffect().setOpacity(1.0)
        QWidget.show(self)
    
    def hide(self, animate=True):
        if not self.isVisible() or self._animFadeOut.state() == QAbstractAnimation.Running:
            return

        if animate:
            self._animFadeOut.start()
        else:
            self.graphicsEffect().setOpacity(0.0)
            self._hide()

    def _hide(self):
        QWidget.hide(self)
        # clear the queues so that they are ready to get filled again for the next show() call
        del self._imageQueue[:]
        self._textQueue.clear()
        self._layoutedTextQueue.clear()
