from PyQt5.Qt import Qt
from PyQt5.QtCore import QPropertyAnimation, QAbstractAnimation, QPoint, QRect, QRectF, QPointF, QSizeF
from PyQt5.QtGui import QColor, QPainter
from PyQt5.QtWidgets import QWidget, QGraphicsOpacityEffect


class _ImageItem():
    def __init__(self, rect, image, align, flags):
        self.rect = rect
        self.image = image
        self.align = align
        self.flags = flags

    def getDrawRect(self, absWidth, absHeight):
        rectWidth = self.rect.width() * absWidth
        rectHeight = self.rect.height() * absHeight
        imageWidth = self.image.width()
        imageHeight = self.image.height()

        x = self.rect.left() * absWidth
        y = self.rect.top() * absHeight

        # image wider then rect -> no horizontal alignment
        if imageWidth >= rectWidth:
            width = rectWidth
        else:
            width = imageWidth
            if self.align & Qt.AlignHCenter:
                x += (rectWidth - imageWidth) / 2

        # image taller than rect -> no vertical alignment
        if imageHeight >= rectHeight:
            height = rectHeight
        else:
            height = imageHeight
            if self.align & Qt.AlignVCenter:
                y += (rectHeight - imageHeight) / 2

        # scale image if aspect ratio need to be kept
        if self.flags & Qt.KeepAspectRatio:
            ratio = imageWidth / float(imageHeight)
            # too tall
            if height * ratio > width:
                height_ = height
                height = int(width / ratio)
                y += (height_ - height) / 2
            # too wide
            else:
                width_ = width
                width = int(height * ratio)
                x += (width_ - width) / 2
        
        return QRect(x, y, width, height)


class _TextItem():
    def __init__(self, rect, text, color, align):
        self.rect = rect
        self.text = text
        self.color = color
        self.align = align

    def getDrawRect(self, absWidth, absHeight):
        x = self.rect.left() * absWidth
        y = self.rect.top() * absHeight
        width = self.rect.width() * absWidth
        height = self.rect.height() * absHeight
        return QRect(x, y, width, height)


class OverlayWidget(QWidget):
    def __init__(self, parent):
        """
        @param parent: the parent of this widget. An overlay without a parent makes no sense
        """
        QWidget.__init__(self, parent)
        
        palette = self.palette()
        palette.setColor(palette.Background, Qt.transparent)
        self.setPalette(palette)

        self._baseColor = None
        self._imageQueue = []
        self._textQueue = {}
        
        opacityEffect = QGraphicsOpacityEffect(self)
        self.setGraphicsEffect(opacityEffect)
        # animation for fade-in
        self._animFadeIn = QPropertyAnimation(opacityEffect, "opacity")
        self._animFadeIn.setDuration(500)
        self._animFadeIn.setStartValue(0.0)
        self._animFadeIn.setEndValue(1.0)
        # animation for fade-out
        self._animFadeOut = QPropertyAnimation(opacityEffect, "opacity", finished=self._hide)
        self._animFadeOut.setDuration(500)
        self._animFadeOut.setStartValue(1.0)
        self._animFadeOut.setEndValue(0.0)

    def paintEvent(self, event):
        painter = QPainter(self)
        painter.setRenderHint(QPainter.Antialiasing)
        painter.fillRect(event.rect(), QColor(*self._baseColor))

        width = self.width()
        height = self.height()

        for imageItem in self._imageQueue:
            # debug:
#             x = imageItem.rect.left() * width
#             y = imageItem.rect.top() * height
#             w = imageItem.rect.width() * width
#             h = imageItem.rect.height() * height
#             painter.drawRect(QRect(x, y, w, h))

            rect = imageItem.getDrawRect(width, height)
            # scale image smoothly
            image = imageItem.image.scaled(rect.width(), rect.height(), transformMode=Qt.SmoothTransformation)
            painter.drawImage(rect.topLeft(), image)

        for textSize, textItems in self._textQueue.items():
            font = painter.font()
            font.setPixelSize(textSize)
            painter.setFont(font)
            fontHeight = painter.fontMetrics().height() - painter.fontMetrics().xHeight()

            for textItem in textItems:
                painter.setPen(QColor(*textItem.color))
                rect = textItem.getDrawRect(width, height)
                flags = textItem.align | Qt.TextDontClip | Qt.TextWordWrap
                
                if rect.size().isValid():
                    #painter.drawRect(rect)  # debug
                    painter.drawText(rect, flags, textItem.text)
                else:
                    pos = rect.topLeft() + QPoint(0, 2 + fontHeight)
                    painter.drawText(pos, textItem.text)

    def _relPos(self, x, y):
        return (x / float(self.width()), y / float(self.height()))

    def getRelativeLocation(self, loc):
        """
        calculates the relative location from the given (absolute) location.
        @param loc: a QRect or QPoint to specify the (absolute) position (and size).
        @return: a QRectF or QPointF specifying the relative position (and size).
        """
        if isinstance(loc, QPoint) or isinstance(loc, QPointF):
            return QPoint(*self._relPos(loc.x(), loc.y()))
        else:
            point = QPointF(*self._relPos(loc.left(), loc.top()))
            size = QSizeF(*self._relPos(loc.width(), loc.height()))
            return QRectF(point, size)

    # TODO: maybe upscaling option?
    def addImage(self, loc, image, align=Qt.AlignCenter, flags=Qt.KeepAspectRatio):
        """
        adds an image at the specified location to the image queue. It will be displayed on the next show() call.
        @param loc: a QRectF or QPointF to specify relative position (and size). If this is a QPoint, the size of the image will be used.
        @param image: the image to add to the image queue.
        @keyword align: (optional) the alignment method for the image, one of Qt's alignment options.
        @keyword flags: (optional) additional flags for the image scaling method.
        """
        if isinstance(loc, QPointF):
            rect = QRectF(loc, image.size())
        else:
            rect = loc

        imageItem = _ImageItem(rect, image, align, flags)
        self._imageQueue.append(imageItem)

    def addText(self, loc, text, textSize=30, color=(0, 0, 0), align=Qt.AlignCenter):
        """
        adds text at the specified location to the text queue. It will be displayed on the next show() call.
        @param loc: a QRectF or QPointF to specify relative position (and size).
        @param text: the String that will be displayed
        @keyword textSize: (optional) the size of the text
        @keyword color: (optional) the text color
        @keyword align: (optional) the alignment of the text inside its rect (only used when loc is a valid QRectF)
        """
        if isinstance(loc, QPointF):
            rect = QRectF(loc.x(), loc.y(), -1, -1)
        else:
            rect = loc
        
        textItem = _TextItem(rect, text, color, align)
        # check if key is not present yet
        if textSize not in self._textQueue:
            self._textQueue[textSize] = []
        self._textQueue[textSize].append(textItem)

    def mousePressEvent(self, event):
        # fade out on click
        if event.button() == Qt.LeftButton:
            # prevent closing before fade-in has finished
            fadeInFlag = self._animFadeIn.state() == QAbstractAnimation.Stopped
            fadeOutFlag = self._animFadeOut.state() == QAbstractAnimation.Stopped
            if fadeInFlag and fadeOutFlag:
                self._animFadeOut.start()

    def show(self, animate=True, color=(150, 150, 150, 150)): #, parent=None):
        if self.isVisible():
            return

        self._baseColor = color

        self._show(animate)
#         if parent is None or parent.isVisible():
#             self._show(animate)
#         else:
#             showEvent_ = parent.showEvent
#             def showEventOverload(*args, **kwargs):
#                 self._show(animate)
#                 showEvent_(*args, **kwargs)
#             parent.showEvent = showEventOverload

    def _show(self, animate):
        if animate:
            self._animFadeIn.start()
        else:
            self.graphicsEffect().setOpacity(1.0)
        QWidget.show(self)

    def hide(self, animate=True):
        if not self.isVisible() or self._animFadeOut.state() == QAbstractAnimation.Running:
            return

        if animate:
            self._animFadeOut.start()
        else:
            self.graphicsEffect().setOpacity(0.0)
            self._hide()

    def _hide(self):
        QWidget.hide(self)
        # clear the queues so that they are ready to get filled again for the next show() call
        del self._imageQueue[:]
        self._textQueue.clear()
