from PyQt5.Qt import QRect, Qt
from PyQt5.QtGui import QPainter, QColor
from PyQt5.QtWidgets import QBoxLayout, QWidget, QVBoxLayout, QSpacerItem

from lib import constants


class ImageWidget(QWidget):
    def __init__(self, parent=None):
        QWidget.__init__(self, parent)
        self.image = None
        
    def setImage(self, image):
        self.image = image

    def paintEvent(self, event):
        painter = QPainter(self)
        if self.image:
            painter.drawImage(self.rect(), self.image) 


class WaveOverlayWidget(QWidget):
    def __init__(self, parent=None):
        QWidget.__init__(self, parent=parent)    
        self.setAutoFillBackground(False)
        self.setAttribute(Qt.WA_TransparentForMouseEvents)

        self._frame = None
        self._state = None
           
    def paintEvent(self, event):
        waveRect = self.parent().settingContainer.getRectForWaves()
        self.setGeometry(waveRect)
        painter = QPainter(self)

        if self._frame:
            #painter.drawImage(self.rect(), self._frame)
            painter.drawImage(self.rect(), self._frame.scaled(self.rect().width(), self.rect().height(), transformMode=Qt.SmoothTransformation))

        if self._state:
            textRect = self.rect().adjusted(3, 3, -3, -3)
            painter.setPen(Qt.NoPen)
            painter.setBrush(QColor(250, 250, 250, 100))
            alignment = Qt.AlignRight | Qt.AlignBottom
            
            bounds = painter.fontMetrics().boundingRect(textRect, alignment, self._state)
            painter.drawRect(bounds.adjusted(-2, -2, 2, 2))
            painter.setPen(QColor(Qt.black))
            painter.drawText(textRect, alignment, self._state)

    def setFrame(self, image):
        self._frame = image
        self.update()

    def setState(self, stateLabel):
        self._state = stateLabel
        self.update()


class SettingContainer(QWidget):
    def __init__(self, emitterPlacement, receiverPlacement, canvas, aspectRatio, parent=None):
        QWidget.__init__(self, parent=parent)

        self.emitterPlacement = emitterPlacement
        self.receiverPlacement = receiverPlacement
        self.canvas = canvas
        self.setLayout(QBoxLayout(QBoxLayout.TopToBottom, parent=self))
        self.layout().setContentsMargins(0, 0, 0, 0)
        self.layout().setSpacing(0)
        
        self._settingLayout = QVBoxLayout()
        self._settingLayout.setContentsMargins(0, 0, 0, 0)
        self._settingLayout.setSpacing(0)
        self._settingLayout.addWidget(emitterPlacement)
        self._settingLayout.addWidget(receiverPlacement, stretch=100 * constants.WATER_TO_MAT_RATIO)
        self._settingLayout.addWidget(canvas, stretch=100.0)
        self.setMinimumSize(700, int(700 / aspectRatio))
        self.layout().addLayout(self._settingLayout)
        self.layout().addItem(QSpacerItem(0, 0))

        self._aspectRatio = aspectRatio

    def resizeEvent(self, event):
        aspectRatio = event.size().width() / float(event.size().height())
        
        # too wide
        if aspectRatio > self._aspectRatio:
            self.layout().setDirection(QBoxLayout.LeftToRight)
            widgetStretch = self.height() * self._aspectRatio
            outerStretch = self.width() - widgetStretch + 0.5
        # too tall
        else:
            self.layout().setDirection(QBoxLayout.TopToBottom)
            widgetStretch = self.width() / self._aspectRatio
            outerStretch = self.height() - widgetStretch + 0.5

        self.layout().setStretch(0, widgetStretch)
        self.layout().setStretch(1, outerStretch)

    def getRectForWaves(self):
        x = self.x() + self.receiverPlacement.x()
        y = self.y() + self.receiverPlacement.y()
        w = self.receiverPlacement.width()
        h = self.receiverPlacement.height() + self.canvas.height()
        
        return QRect(x, y, w, h)

    def getCenterXEmitRec(self):
        centerXEmit = self.emitterPlacement.getCenterX()
        centerXRec = self.receiverPlacement.getCenterX()
        
        return (centerXEmit, centerXRec)
        
    def setCenterXOfEmitRec(self, centerXEmitRec):
        centerXEmit, centerXRec = centerXEmitRec
        self.emitterPlacement.setCenterX(centerXEmit)
        self.receiverPlacement.setCenterX(centerXRec)

    def setMaterialImage(self, image):
        self.canvas.setImage(image)
        self.canvas.update()
