from PyQt5.QtCore import QSize, Qt
from PyQt5.QtGui import QIcon, QImage, QPainter
from PyQt5.QtWidgets import QAction, QActionGroup, QGridLayout, QPushButton, QSizePolicy, QSpacerItem, QToolBar, QWidget

import instance
from lib import constants
from lib.constants import ResPath
from lib.i18n import lu
from util import renderutil
from widgets.materialeditor.canvas import Canvas
from widgets.materialeditor.emitterplacement import EmitterPlacement
from widgets.materialeditor.gallerywidget import GalleryWidget
from widgets.materialeditor.receiverplacement import ReceiverPlacement
from widgets.materialeditor.settingcontainer import SettingContainer


class MaterialEditorWidget(QWidget):
    def __init__(self, parent=None):
        QWidget.__init__(self, parent)

        # app layout
        gridLayout = QGridLayout(self)
        gridLayout.setVerticalSpacing(0)  # we don't want vertical space between the widgets

        # emitter / receivers widget + water
        self._emitterPlacement = EmitterPlacement()
        self._receiverPlacement = ReceiverPlacement()
        # canvas
        self._canvas = Canvas(instance.computationManager().getProcess().materialSize())
        # create the aspect ratio container
        self.settingContainer = SettingContainer(self._emitterPlacement, self._receiverPlacement, self._canvas, instance.getMaterialAspectRatio())
        self.settingContainer.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Expanding)
        gridLayout.addWidget(self.settingContainer, 0, 1, 2, 2)

        # Toolbar
        toolbar = QToolBar(orientation=Qt.Vertical)
        toolbar.setIconSize(QSize(32, 32))
        gridLayout.addWidget(toolbar, 1, 0)
        # actions for draw mode
        penGroup = QActionGroup(self)
        self._aMaterial = QAction(QIcon(ResPath.icons + "rock.png"), lu("actionSwitchToMaterial"), penGroup, triggered=self._canvas.activatePencil, checkable=True, checked=True)
        QAction(QIcon(ResPath.icons + "eraser.png"), lu("actionSwitchToEraseMode"), penGroup, shortcut="E", triggered=lambda: self._canvas.setMode(Canvas.MODE_ERASER), checkable=True)
        QAction(QIcon(ResPath.icons + "fill.png"), lu("actionSwitchToFillMode"), penGroup, shortcut="F", triggered=lambda: self._canvas.setMode(Canvas.MODE_FILL), checkable=True)
        toolbar.addActions(penGroup.actions())
        # clear action
        toolbar.addAction(QAction(QIcon(ResPath.icons + "clear.png"), lu("actionClearDrawingBoard"), self, shortcut="C", triggered=self._canvas.clear))

        # space below canvas
        gridLayout.addItem(QSpacerItem(0, 10), 2, 0, 1, 3)
        # example gallery
        self._gallery = GalleryWidget(title=lu("labelExamples") + ":", thumbCallback=self._canvas.setImage)
        self._gallery.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self._gallery.setMinimumSize(100, 225)
        self._gallery.setAutoFillBackground(True)
        gridLayout.addWidget(self._gallery, 3, 1, 1, 2)

        # space below examples
        gridLayout.addItem(QSpacerItem(0, 10), 4, 0, 1, 2)
        # help  button
        bHelp = QPushButton("?", clicked=self.showHelp)
        bHelp.setMaximumWidth(toolbar.sizeHint().width())
        gridLayout.addWidget(bHelp, 5, 0)
        # start simulation button
        bStartSim = QPushButton(lu("buttonStartSimulation"), clicked=self.onStartSimulation)
        bStartSim.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        gridLayout.addWidget(bStartSim, 5, 1)
        # wave config button
#         self._bWaveConfig = QPushButton(icon=QIcon(ResPath.icons + "gear.png"), clicked=self.openWaveDialog, enabled=False)
#         self._bWaveConfig.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
#         gridLayout.addWidget(self._bWaveConfig, 5, 2)

    def onStartSimulation(self):
        # stop inversion
        window = instance.window()
        window._inversionWidget.reset()
        
        # TODO: save RGB image, instead of rendered image
        canvasImage = QImage(self._canvas.size(), QImage.Format_ARGB32)
        self._canvas.render(QPainter(canvasImage))
        # save current drawn configuration
        if constants.DEBUG:
            canvasImage.save("../tmp/materialConfig.png")
        self._gallery.addLatestUserMaterial(canvasImage)
        
        material = renderutil.qImageToArray(self._canvas.getRGBImage(), self._canvas.getMaterialColor())

        emitterPosition = self._emitterPlacement.relEmitterPosition()
        receiverPositions = self._receiverPlacement.relReceiversPositions()
        centerXEmitRec = self.settingContainer.getCenterXEmitRec()

        # switch to simulation tab
        window.switchToSimulation(material, emitterPosition, receiverPositions, centerXEmitRec)

    def openWaveDialog(self):
        from widgets.signalconfig.waveconfiguration import WaveDialog
        d = WaveDialog(self)
        if d.exec_():
            pass
            # that is the resulting function:
            # TODO use this function
#             instance.context().sourceFunction = d.getResult()

    def showHelp(self):
        overlay = instance.window().getOverlay()
        rect = overlay.getRelativeLocation(self.childrenRect())
        # TODO: weird behaviour of pos
        print self.pos(), self.childrenRect()
        overlay.addText(rect, "TODO")
        overlay.show()

    def reset(self):
        self._canvas.clear()
        # set hard material pencil as current tool
        self._aMaterial.setChecked(True)
        self._aMaterial.trigger()
        # disable wave config button
#         self._bWaveConfig.setEnabled(False)
        # maybe reset settingcontainers
