from datetime import datetime
import os

from PyQt5.QtCore import QRect
from PyQt5.QtGui import QColor, QImage, QPainter
from PyQt5.QtWidgets import QLabel, QWidget

from lib.constants import ResPath
from lib.i18n import lu
from util import renderutil


class GalleryWidget(QWidget):
    def __init__(self, parent=None, title="", thumbHeight=70, thumbCallback=None):
        QWidget.__init__(self, parent)

        self._lTitle = QLabel(title, parent=self)
        self._thumbCallback = thumbCallback
        self.setMouseTracking(True)

        self._thumbTemplates = []
        for fileName in os.listdir(ResPath.examples):
            self._thumbTemplates.append(_Thumb(ResPath.examples + fileName))
        
        # TODO better path
        path = os.path.join(ResPath.tmp, "examples", "")
        # create directory if it doesn't exist
        if not os.path.exists(path):
            os.makedirs(path)
        files = [os.path.join(path, fileName) for fileName in os.listdir(path)]
        files.sort(key=os.path.getmtime, reverse=True)
        self._thumbLatest = []
        for filePath in files:
            self._thumbLatest.append(_Thumb(filePath, showLabel=False))

        self._thumbHeight = thumbHeight
        self._spacing = 8

    def setSpacing(self, spacing):
        self._spacing = spacing

    def addLatestUserMaterial(self, material):
        # TODO better path
        path = os.path.join(ResPath.tmp, "examples", "")
        files = os.listdir(path)
        
        curHash = renderutil.qImageToHash(material)
        for thumb in self._thumbLatest:
            if thumb.hash == curHash:
                print "already present"
                os.utime(thumb.filePath, None)
                return

        # only save the latest 5 materials        
        if len(files) >= 5:
            files = [os.path.join(path, fileName) for fileName in files]
            # sort files by modification time (latest last)
            files.sort(key=os.path.getmtime)
            # delete oldest material
            os.remove(files[0])
        
        fileName = datetime.now().strftime("%Y%m%d-%H%M%S") + ".png"
        material.save(path + fileName)
        
        # TODO use Queue or sth. similar
        files = [os.path.join(path, fileName) for fileName in os.listdir(path)]
        files.sort(key=os.path.getmtime, reverse=True)
        self._thumbLatest = []
        for filePath in files:
            self._thumbLatest.append(_Thumb(filePath, showLabel=False))
        self.adjustThumbs()

    def adjustThumbs(self):
        fontMetrics = self.fontMetrics()
        x = self._spacing
        y = self._lTitle.height() + self._spacing
        addY = self._thumbHeight + self._spacing
        latestY = self.height() - (self._thumbHeight + self._spacing)

        for thumb in self._thumbTemplates:
            width = thumb.getWidthForHeight(self._thumbHeight, fontMetrics)
            addX = width + self._spacing

            if x + addX > self.width():
                if y + 2*addY <= latestY:
                    x = self._spacing
                    y += addY
                else:
                    break

            thumb.setGeometry(x, y, width, self._thumbHeight)
            x += addX

        x = self._spacing
        widthFlag = False
        for thumb in self._thumbLatest[:]:
            if not widthFlag:
                width = thumb.getWidthForHeight(self._thumbHeight, fontMetrics)
                addX = width + self._spacing
    
                if x + addX > self.width():
                    widthFlag = True
                    self._thumbLatest.remove(thumb)
    
                thumb.setGeometry(x, latestY, width, self._thumbHeight)
                x += addX
            else:
                self._thumbLatest.remove(thumb)
        
        self.update()

    def resizeEvent(self, event):
        QWidget.resizeEvent(self, event)
        self.adjustThumbs()

    def paintEvent(self, event):
        QWidget.paintEvent(self, event)
        painter = QPainter(self)
        rect = event.rect()
        
        for thumb in self._thumbTemplates:
            if rect.intersects(thumb.rect()):
                thumb.paint(painter)

        for thumb in self._thumbLatest:
            if rect.intersects(thumb.rect()):
                thumb.paint(painter)

    def mousePressEvent(self, event):
        QWidget.mousePressEvent(self, event)
        if self._thumbCallback:
            for thumb in self._thumbTemplates + self._thumbLatest:
                if thumb.rect().contains(event.pos()):
                    self._thumbCallback(thumb.image.copy())
    
    def mouseMoveEvent(self, event):
        QWidget.mouseMoveEvent(self, event)
        
        flag = False
        for thumb in self._thumbTemplates + self._thumbLatest:
            if thumb.rect().contains(event.pos()):
                if not thumb.underMouse:
                    flag = True
                thumb.underMouse = True
            else:
                if thumb.underMouse:
                    flag = True
                thumb.underMouse = False
        
        if flag:
            self.update()

class _Thumb():
    def __init__(self, filePath, showLabel=True):
        self.filePath = filePath
        self.image = QImage(filePath)
        self.hash = renderutil.qImageToHash(self.image)
        
        if showLabel:
            self.name = os.path.basename(filePath)
            self.name = lu("example " + self.name[:self.name.rfind(".")])
        else:
            self.name = None
        self.setGeometry(-1, -1, 0, 0)
        self.underMouse = False

    def setGeometry(self, x, y, width, height):
        self._rect = QRect(x, y, width, height)

    @property
    def x(self):
        return self._rect.x()

    @property
    def y(self):
        return self._rect.y()

    @property
    def width(self):
        return self._rect.width()

    @property
    def height(self):
        return self._rect.width()
    
    def rect(self):
        return self._rect

    def getWidthForHeight(self, height, fontMetrics):
        if self.name:
            fontHeight = fontMetrics.height()
            ratio = float(self.image.width()) / (self.image.height() + fontHeight)
        else:
            ratio = float(self.image.width()) / self.image.height()
        return int(ratio * height)

    def paint(self, painter):
        imageRect = self._rect
        
        if self.name:
            fontHeight = painter.fontMetrics().height()
            imageRect = imageRect.adjusted(0, 0, 0, -fontHeight)

        painter.drawImage(imageRect, self.image)
        
        if self.name:
            nameWidth = painter.fontMetrics().width(self.name)
            textX = self.x + (self.width - nameWidth) / 2.0
            textY = self._rect.bottom() - painter.fontMetrics().descent()
            painter.drawText(textX, textY, self.name)
        
        # mouse hover highlighting
        if self.underMouse:
            painter.setOpacity(0.2)
            painter.fillRect(imageRect, QColor(100, 255, 255))
            painter.setOpacity(1.0)
