from PyQt5.QtCore import Qt, QPoint
from PyQt5.QtGui import QPainter, QColor
from PyQt5.QtWidgets import QWidget

from lib.constants import Orientation
from util import utils


class DragObject(QWidget):
    """
    a moveable object to drag around
    """
    def __init__(self, parent=None, xrel=0, orientation=Orientation.BOTH):
        QWidget.__init__(self, parent=parent)

        self.xrel = xrel
        self._orientation = orientation
        self._image = None
        self._lastPos = QPoint()
        self.setDraggable(True)

    def setImage(self, image):
        self._image = image.smoothScaled(self.width(), self.height())

    def setDraggable(self, draggable):
        self._draggable = draggable
        if self._draggable:
            self.setCursor(Qt.OpenHandCursor)
        else:
            self.unsetCursor()

    def paintEvent(self, event):
        QWidget.paintEvent(self, event)
        painter = QPainter(self)
        if self._image:
            painter.drawImage(self.rect(), self._image)
        else:
            painter.fillRect(self.rect(), QColor(0, 0, 0))

    def mousePressEvent(self, event):
        if self._draggable and event.button() == Qt.LeftButton:
            self._lastPos = event.pos()
            self.setCursor(Qt.ClosedHandCursor)

    def mouseMoveEvent(self, event):
        if self._draggable and event.buttons() & Qt.LeftButton and self._lastPos:
            targetX, targetY = self.pos().x(), self.pos().y()
            # prevent to move this outside of the parent
            if self._orientation & Orientation.HORIZONTAL:
                targetX += event.pos().x() - self._lastPos.x()
                targetX = utils.clamp(targetX, 0, self.parent().width() - self.width())
            if self._orientation & Orientation.VERTICAL:
                targetY += event.pos().y() - self._lastPos.y()
                targetY = utils.clamp(targetY, 0, self.parent().height() - self.height())

            self.xrel = (2 * targetX + self.width()) / (2.0 * self.parent().width())
            self.move(targetX, targetY)

    def mouseReleaseEvent(self, event):
        if self._draggable and event.button() == Qt.LeftButton and self._lastPos:
            self._lastPos = QPoint()
            self.setCursor(Qt.OpenHandCursor)
