from PIL import ImageDraw
from PIL.ImageQt import ImageQt
from PyQt5.QtCore import QPoint, QRect, Qt
from PyQt5.QtGui import QColor, QCursor, QImage, QPainter, QPen, QPixmap
from PyQt5.QtWidgets import QWidget

from lib.constants import ResPath, COLOR_MATERIAL, COLOR_BACKGROUND
from util import renderutil


class Canvas(QWidget):
    """
    a canvas to draw onto
    """
    MODE_PENCIL = 0
    MODE_ERASER = 1
    MODE_FILL = 2

    def __init__(self, imageSize, parent=None, bgColor=QColor(*COLOR_BACKGROUND), matColor=QColor(*COLOR_MATERIAL)):
        """
        creates a new graphics_canvas widget
        @param parent the parent of this widget
        @param bgColor the background color
        @keyword matColor: the material color
        """
        super(QWidget, self).__init__(parent)

        # colors
        self._bgColor = bgColor
        self._matColor = matColor
        # cursors
        self._cursors = [
            QCursor(QPixmap(ResPath.icons + "pencil.png").scaled(24, 24, transformMode=Qt.SmoothTransformation), hotX=0, hotY=24),
            QCursor(QPixmap(ResPath.icons + "eraser.png").scaled(24, 24, transformMode=Qt.SmoothTransformation)),
            QCursor(QPixmap(ResPath.icons + "fill.png").scaled(24, 24, transformMode=Qt.SmoothTransformation), hotX=4, hotY=18)
        ]
        self.setCursor(self._cursors[Canvas.MODE_PENCIL])
        # pen, eraser, mode
        self._pen = QPen(matColor, 4, style=Qt.SolidLine, cap=Qt.RoundCap, join=Qt.RoundJoin)
        self._eraser = QPen(bgColor, 10, style=Qt.SolidLine, cap=Qt.RoundCap, join=Qt.RoundJoin)
        self._mode = Canvas.MODE_PENCIL
        # background image
        self._image = QImage(imageSize[0], imageSize[1], QImage.Format_ARGB32)
        self._image.fill(self._bgColor)

        self._lastPos = QPoint()

    def setImage(self, image):
        self._image = image.scaled(self._image.width(), self._image.height())
        self.update()

    def clear(self):
        self._image.fill(self._bgColor)
        self.update()

    def getEmptyRGBImage(self):
        img = self._image.copy()
        img.fill(self._bgColor)
        return img.convertToFormat(QImage.Format_RGB32)

    def getRGBImage(self):
        return self._image.convertToFormat(QImage.Format_RGB32)

    def getMaterialColor(self):
        return self._matColor.getRgb()[:3]

    # TODO: remove this when we definitely don't use other materials
    def activatePencil(self):
        self._pen.setColor(self._matColor)
        self.setMode(Canvas.MODE_PENCIL)

    def setMode(self, mode):
        self._mode = mode
        self.setCursor(self._cursors[mode])

    def paintEvent(self, event):
        painter = QPainter(self)
#         painter.drawImage(0, 0, self._image.scaled(self.width(), self.height()))
        painter.drawImage(event.rect(), self._image.scaled(self.width(), self.height()), event.rect())

        if self._lastPos and self._mode == Canvas.MODE_ERASER:
            radX, radY = self.mapFromSrc(self._eraser.width(), self._eraser.width())
            radX /= 2.0
            radY /= 2.0
            painter.drawEllipse(self._lastPos, radX, radY)

    def mousePressEvent(self, event):
        if event.button() == Qt.LeftButton:
            if self._mode == Canvas.MODE_PENCIL or self._mode == Canvas.MODE_ERASER:
                self._lastPos = event.pos()
            elif self._mode == Canvas.MODE_FILL:
                self._floodFill(event.pos(), self._pen.color().getRgb())

    def mouseMoveEvent(self, event):
        QWidget.mouseMoveEvent(self, event)
        if event.buttons() & Qt.LeftButton and self._lastPos:
            # when the mouse moved enough, draw a line
            if self._mode == Canvas.MODE_PENCIL and (event.pos() - self._lastPos).manhattanLength() > min(self.mapFromSrc(self._pen.width(), self._pen.width())) / 2.0:
                self.drawLineTo(event.pos(), self._pen)
            elif self._mode == Canvas.MODE_ERASER:
                self.drawLineTo(event.pos(), self._eraser)

    def mouseReleaseEvent(self, event):
        QWidget.mouseReleaseEvent(self, event)
        if event.button() == Qt.LeftButton and self._lastPos:
            if self._mode == Canvas.MODE_PENCIL:
                self.drawLineTo(event.pos(), self._pen)
            elif self._mode == Canvas.MODE_ERASER:
                self.drawLineTo(event.pos(), self._eraser)
            self._lastPos = QPoint()

    def mapToSrc(self, point):
        xRatio = float(self._image.width()) / self.width()
        yRatio = float(self._image.height()) / self.height()
        return QPoint(point.x() * xRatio, point.y() * yRatio)

    def mapFromSrc(self, width, height):
        xRatio = self.width() / float(self._image.width())
        yRatio = self.height() / float(self._image.height())
        return (width * xRatio, height * yRatio)

    def drawLineTo(self, pos, pen):
        srcImgLastPos = self.mapToSrc(self._lastPos)
        srcImgPos = self.mapToSrc(pos)

        painter = QPainter(self._image)
        painter.setPen(pen)
        if pos == self._lastPos:
            painter.drawPoint(srcImgPos)
        else:
            painter.drawLine(srcImgLastPos, srcImgPos)

        xAdj, yAdj = self.mapFromSrc(pen.width(), pen.width())
        xAdj = xAdj * 3.0 / 4.0
        yAdj = yAdj * 3.0 / 4.0
        self.update(QRect(self._lastPos, pos).normalized().adjusted(-xAdj, -yAdj, xAdj, yAdj))
        # update last mouse pos
        self._lastPos = pos

    def _floodFill(self, pos, replaceColor):
        p = self.mapToSrc(pos)
        pilImage = renderutil.qImageToPillow(self._image)
        ImageDraw.floodfill(pilImage, (p.x(), p.y()), replaceColor)
        self._image = ImageQt(pilImage)
        self.update()
