from PyQt5.QtCore import QSize, Qt, QRectF
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QGridLayout, QPushButton, QSizePolicy, QSpacerItem, QToolBar, QWidget

import instance
from lib import constants
from lib.constants import MAX_INV_ITERATIONS, ResPath
from lib.i18n import lu
import numpy as np
from util import renderutil
from widgets.inversion.seismogramwidget import SeismogramView
from widgets.materialeditor.emitterplacement import EmitterPlacement
from widgets.materialeditor.receiverplacement import ReceiverPlacement
from widgets.materialeditor.settingcontainer import ImageWidget, SettingContainer, WaveOverlayWidget


class InversionWidget(QWidget):
    def __init__(self, parent=None):
        QWidget.__init__(self, parent)

        # main layout
        gridLayout = QGridLayout(self)
        gridLayout.setVerticalSpacing(0)  # we don't want vertical space between the widgets

        #=======================================================================
        # settings widget
        #=======================================================================
        self._emitterPlacement = EmitterPlacement()
        self._emitterPlacement.setDraggable(False)
        self._receiverPlacement = ReceiverPlacement()
        self._receiverPlacement.setDraggable(False)
        # create the aspect ratio container
        self.settingContainer = SettingContainer(self._emitterPlacement, self._receiverPlacement, ImageWidget(), instance.getMaterialAspectRatio())
        self.settingContainer.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Expanding)
        gridLayout.addWidget(self.settingContainer, 0, 1)        
        self.waveWidget = WaveOverlayWidget(self)
        
        # spacing between canvas and seismogram
        gridLayout.addItem(QSpacerItem(0, 10, vPolicy=QSizePolicy.Fixed), 1, 0, 1, 2)
    
        #=======================================================================
        # toolbar
        #=======================================================================
        toolbar = QToolBar(orientation=Qt.Vertical)
        toolbar.setLayoutDirection(Qt.RightToLeft)
        toolbar.setIconSize(QSize(32, 32))
        # stop action
        #toolbar.addAction(QAction(QIcon(ResPath.icons + "stop.png"), lu("actionStop"), self, triggered=self.stop))
        
        # play/pause action
        icon = QIcon(ResPath.icons + "play.png")
        icon.addFile(ResPath.icons + "pause.png", state=QIcon.On)
        trigger = lambda: None  # TODO
        self._aPlayPause = QAction(icon, lu("actionPlay"), self, checkable=True, checked=True, shortcut=" ", triggered=trigger)
        toolbar.addAction(self._aPlayPause)

        gridLayout.addWidget(toolbar, 2, 0, alignment=Qt.AlignBottom)
        
        
        # "real material"-button
        self._buttonRealMaterial = QPushButton("\n".join(lu("buttonShowRealMaterial")), pressed=self.real_material_pressed, released=self.real_material_released)
        self._buttonRealMaterial.setMaximumWidth(toolbar.sizeHint().width())
        self._buttonRealMaterial.setFocusPolicy(Qt.NoFocus)
        gridLayout.addWidget(self._buttonRealMaterial, 0, 0)

        # seismogram view
        self._seismogram = SeismogramView()
        gridLayout.addWidget(self._seismogram, 2, 1)
        # space below seismogram
        gridLayout.addItem(QSpacerItem(0, 10), 3, 0, 1, 2)


        # button for continuing the inversion
        self._buttonContinueInversion = QPushButton(lu("buttonContinueInversion"), clicked=self.continueInversion, enabled=False)
        gridLayout.addWidget(self._buttonContinueInversion, 4, 1)

        self._currentFrame = 0
        self._showOverlayOnNextShow = False

        #=======================================================================
        # initialize inversion variables
        #=======================================================================
        self._iteration = 0
        self._materialImage = None
        self.updateState("labelInitializing")
        
        self._timerId = None
        self.data = {"type": "next_data"}   
        self.frames = []
    
    def real_material_pressed(self):
        if self.data["type"] == "forwardOnly":
            return

        self.settingContainer.setMaterialImage(self._realMaterialImage)
        self.update()
        
    def real_material_released(self):
        if self.data["type"] == "forwardOnly":
            return

        self.settingContainer.setMaterialImage(self._materialImage)       
        self.update()

    def resizeEvent(self, event):
        QWidget.resizeEvent(self, event)
        waveRect = self.settingContainer.getRectForWaves()
        self.waveWidget.setGeometry(waveRect)
    
    def process_new_data(self):
        if self._queue.empty():
            return            
        
        self._currentFrame = 0
        self._seismogram.setCurrentFrame(0)
        self.data = self._queue.get()
        state = self.data["type"]
        print "Got data:", self.data["type"]," Precalucated Data:", self._queue.qsize()

        if state == "forward":                              
            self._iteration += 1
            self.updateState("labelForwardSimulation")                 
            self._seismogram.setForward(True)               
            self.frames = self.data["frames"]
        elif state == "backward":
            self._seismogram.setForward(False)
            self.updateState("labelBackwardSimulation")
            self.frames = self.data["frames"]  
        elif state == "materialUpdate":
            self._materialImage = self.data["material"]        
            self.settingContainer.setMaterialImage(self._materialImage)
     
            if self._iteration == MAX_INV_ITERATIONS:
                if self._timerId:
                    self.killTimer(self._timerId)
                    self._timerId = None
                self.inversionFinished()
                self.data["type"] = "finished"
                return  
            self.data["type"] = "next_data" 
    
    def timerEvent(self, event):  
        if not self._aPlayPause.isChecked():
            return
        
        state = self.data["type"]
        
        if state == "next_data":
            self.process_new_data()
            return

        # frame not ready jet
        if self._currentFrame + 1 >= len(self.frames):
            return
        
        self.advanceFrame()

        if state == "forwardOnly":
            self._seismogram.updateCurrentSeismoData(self.frames[-1][2])
            if self._currentFrame == constants.FRAME_COUNT - 1:
                self.data["type"] = "next_data"
                self.settingContainer.setMaterialImage(self._materialImage)
                self._seismogram.updateOriginalSeismoData(self.frames[-1][2])
                    
        elif state == "forward":
            self._seismogram.updateCurrentSeismoData(self.frames[-1][3])
            if self._currentFrame == constants.FRAME_COUNT - 1:
                self.data["type"] = "next_data"

        elif state == "backward":
            if self._currentFrame == constants.FRAME_COUNT - 1:
                self.data["type"] = "next_data"            

    def advanceFrame(self):
        self._currentFrame += 1
        self._seismogram.setCurrentFrame(self._currentFrame)
        self.waveWidget.setFrame(self.frames[self._currentFrame][0])

    def updateState(self, state):
        if state == "labelInversionFinished" or state == "labelForwardSimulationOnly":
            stateLabel = lu(state)
        else:
            stateLabel = "{}. {}: {}/{}".format(lu(state), lu("labelCurrentIteration"), self._iteration, constants.MAX_INV_ITERATIONS)
        self.waveWidget.setState(stateLabel)

    def inversionFinished(self):
        self.waveWidget.setFrame(None)
        self.updateState("labelInversionFinished")
        self._buttonContinueInversion.setEnabled(True)
        self._emitterPlacement.setDraggable(True)
        self._receiverPlacement.setDraggable(True)
        
        # show overlay message
        ov = instance.window().getOverlay()
        ov.addText(QRectF(0.075, 0.23, 0.55, 0.5), lu("overlayMotivateSettingChange"))
        if self.isVisible():
            ov.show()
        else:
            self._showOverlayOnNextShow = True

    def loadFromData(self, realMaterial, seismoData, centerXEmitRec):
        self.settingContainer.setCenterXOfEmitRec(centerXEmitRec)
        self._seismogram.updateOriginalSeismoData(seismoData)
        self._seismogram.updateCurrentSeismoData(None)

        self._realMaterial = realMaterial
        self._realMaterialImage = renderutil.arrayToQImage(realMaterial)

        # start inversion
        # TODO: Here we should perturb the material and give it as initial
        initialMaterial = np.zeros_like(realMaterial)
        self._queue = instance.computationManager().solveInversion(initialMaterial=initialMaterial)

        # At this point the material is empty all the time? 
        self._materialImage = renderutil.arrayToQImage(initialMaterial, amplify=True)
        self.settingContainer.setMaterialImage(self._materialImage)
        
        self._timerId = self.startTimer(40)

    def reset(self):
        if self._timerId:
            self.killTimer(self._timerId)
            self._timerId = None

        self._buttonContinueInversion.setEnabled(False)
        self._emitterPlacement.setDraggable(False)
        self._receiverPlacement.setDraggable(False)
   
        # TODO kill processes
        self.updateState("labelInitializing")

        # reset counters and data
        self._iteration = 0
        self._currentFrame = 0
        self._seismogram.setCurrentFrame(0)
        self._showOverlayOnNextShow = False
        self.data = {"type": "next_data"}   
        self.frames = []

    def continueInversion(self):
        self._buttonContinueInversion.setEnabled(False)
        self._emitterPlacement.setDraggable(False)
        self._receiverPlacement.setDraggable(False)

        emitterPosition = self._emitterPlacement.relEmitterPosition()
        receiverPositions = self._receiverPlacement.relReceiversPositions()

        self.reset()

        frames = instance.computationManager().solveForward(self._realMaterial, emitterPosition, receiverPositions)
        self.data = {"type": "forwardOnly", "frames": frames}
        
        # prepare state for forward only
        self.updateState("labelForwardSimulationOnly")
        self._seismogram.setForward(True) 
        self.frames = self.data["frames"]
        self.settingContainer.setMaterialImage(self._realMaterialImage)
        self._seismogram.updateOriginalSeismoData(None)

        self._timerId = self.startTimer(40)

        # queue the request for an inversion solve
        self._queue = instance.computationManager().solveInversion()

    def showEvent(self, event):
        if self._showOverlayOnNextShow:
            instance.window().getOverlay().show()
            self._showOverlayOnNextShow = False
