if __name__ == "__main__":
    import os
    os.chdir("../")

from PyQt5.QtCore import Qt, QSize
from PyQt5.QtGui import QGuiApplication, QIcon, QImage, QPalette, QPixmap
from PyQt5.QtWidgets import QAction, QDialog, QLabel, QScrollArea, QSizePolicy, QToolBar, QVBoxLayout
from lib.constants import ResPath
from util import utils
from lib.i18n import lu

class ImageViewerDialog(QDialog):
    def __init__(self, image, parent=None, title="ImageViewer"):
        QDialog.__init__(self, parent)
        self.setWindowTitle(title)
        self.setWindowModality(Qt.ApplicationModal)
        self.setWindowFlags(Qt.Window)
        
        if image.isNull():
            raise ValueError("No Image provided")

        # create layout
        self.setLayout(QVBoxLayout())
        
        # toolbar
        toolbar = QToolBar()
        toolbar.setIconSize(QSize(32, 32))
        self.layout().addWidget(toolbar, alignment=Qt.AlignCenter)
        # zoom in
        self._aZoomIn = QAction(QIcon(ResPath.icons + "zoomIn.png"), lu("actionZoomIn"), self, triggered=self.zoomIn)
        toolbar.addAction(self._aZoomIn)
        # zoom out
        self._aZoomOut = QAction(QIcon(ResPath.icons + "zoomOut.png"), lu("actionZoomOut"), self, triggered=self.zoomOut)
        toolbar.addAction(self._aZoomOut)
        toolbar.addSeparator()
        # fit to window
        self._aFitToWindow = QAction(QIcon(ResPath.icons + "fitToWindow.png"), lu("actionFitToWindow"), self, triggered=self.fitToWindow)
        toolbar.addAction(self._aFitToWindow)
        # contain to window
        self._aContainToWindow = QAction(QIcon(ResPath.icons + "containInWindow.png"), lu("actionContainInWindow"), self, triggered=self.containInWindow)
        toolbar.addAction(self._aContainToWindow)

        # variables for zooming
        self._minScaleFactor = 0.33
        self._maxScaleFactor = 1.0
        self._scaleFactor = 1.0
        # label for the image
        self._imageLabel = QLabel()
        self._imageLabel.setBackgroundRole(QPalette.Base)
        self._imageLabel.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Ignored)
        self._imageLabel.setScaledContents(True)
        self._imageLabel.setPixmap(QPixmap.fromImage(image))
        self._imageLabel.adjustSize()
        # scroll area that contains the image-label
        self._scrollArea = QScrollArea()
        self._scrollArea.setBackgroundRole(QPalette.Dark)
        self._scrollArea.setAlignment(Qt.AlignCenter)
        self._scrollArea.setWidget(self._imageLabel)
        self.layout().addWidget(self._scrollArea)
        
        self.resize(QGuiApplication.primaryScreen().availableSize() * 7 / 8)

    def resizeEvent(self, event):
        QDialog.resizeEvent(self, event)

        imWidth, imHeight = self._imageLabel.pixmap().width(), self._imageLabel.pixmap().height()
        scWidth, scHeight = self._scrollArea.width(), self._scrollArea.height()
        scaleWidth = (scWidth - 2) / float(imWidth)
        
        if scaleWidth * imHeight < scHeight:
            self._minScaleFactor = scaleWidth
        else:
            self._minScaleFactor = (scHeight - 2) / float(imHeight)
        self._maxScaleFactor = self._minScaleFactor * 4.0
        # asjust scale factor and image label
        self._scaleFactor = utils.clamp(self._scaleFactor, self._minScaleFactor, self._maxScaleFactor)
        self._imageLabel.resize(self._scaleFactor * self._imageLabel.pixmap().size())

    def zoomIn(self):
        self.scaleImage(1.25)

    def zoomOut(self):
        self.scaleImage(0.8)

    def fitToWindow(self):
        imWidth, imHeight = self._imageLabel.pixmap().width(), self._imageLabel.pixmap().height()
        scWidth, scHeight = self._scrollArea.width(), self._scrollArea.height()
        
        width = scWidth - (self._scrollArea.verticalScrollBar().width() + 2)
        self._scaleFactor = width / float(imWidth)
        height = imHeight * self._scaleFactor
        # height to small
        if height < scHeight:
            height = scHeight - (self._scrollArea.horizontalScrollBar().height() + 2)
            self._scaleFactor = height / float(imHeight)
            width = imWidth * self._scaleFactor
        
        self._imageLabel.resize(width, height)
        self.validateActions()

    def containInWindow(self):
        imWidth, imHeight = self._imageLabel.pixmap().width(), self._imageLabel.pixmap().height()
        scWidth, scHeight = self._scrollArea.width(), self._scrollArea.height()
        
        width = scWidth - 2
        self._scaleFactor = width / float(imWidth)
        height = imHeight * self._scaleFactor
        # height to big
        if height > scHeight:
            height = scHeight - 2
            self._scaleFactor = height / float(imHeight)
            width = imWidth * self._scaleFactor
        
        self._imageLabel.resize(width, height)
        self.validateActions()

    def scaleImage(self, factor):
        self._scaleFactor *= factor
        self.validateActions()
        self._scaleFactor = utils.clamp(self._scaleFactor, self._minScaleFactor, self._maxScaleFactor)

        self._imageLabel.resize(self._scaleFactor * self._imageLabel.pixmap().size())

        self.adjustScrollBar(self._scrollArea.horizontalScrollBar(), factor)
        self.adjustScrollBar(self._scrollArea.verticalScrollBar(), factor)

    def validateActions(self):
        self._aZoomIn.setEnabled(self._scaleFactor < self._maxScaleFactor)
        self._aZoomOut.setEnabled(self._scaleFactor > self._minScaleFactor)

    def adjustScrollBar(self, scrollBar, factor):
        scrollBar.setValue(int(factor * scrollBar.value() + ((factor - 1) * scrollBar.pageStep()/2)))


if __name__ == "__main__":
    import sys
    from PyQt5.QtWidgets import QApplication
    
    sys.excepthook = utils.excepthook
    app = QApplication(sys.argv)
    
    image = QImage("../res/doc/en_us/introduction.png")
    d = ImageViewerDialog(image)
    d.show()
    d.containInWindow()
    sys.exit(app.exec_())
