from PyQt5.Qt import QTabWidget
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QIcon, QImage, QPalette, QPixmap, QPainter
from PyQt5.QtWidgets import QGridLayout, QLabel, QPushButton, QScrollArea, QVBoxLayout, QWidget

from lib.constants import ResPath
from lib.i18n import lu, luDir
from widgets.imageviewer import ImageViewerDialog


class DocumentationTabWidget(QTabWidget):
    def __init__(self, handle, parent=None):
        QTabWidget.__init__(self, parent)
        # tutorial documentation
        docContextTut = DocumentationContext()
        self._docWidgetTut = DocumentationWidget(docContextTut,parent=self)
        # math documentation
        docContextMath = DocumentationContext()
        self._docWidgetMath = DocumentationWidget(docContextMath,parent=self)

        # factory functions for switching tabs and page
        def gotoTut(index=None):
            def func():
                self.setCurrentIndex(0)
                if index is not None:
                    self._docWidgetTut.goTo(index)
            return func
        def gotoMath(index=None):
            def func():
                self.setCurrentIndex(1)
                if index is not None:
                    self._docWidgetMath.goTo(index)
            return func

        #=======================================================================
        # supply tutorial context
        #=======================================================================
        # introduction
        docContextTut += "introduction.png", (lu("buttonChooseASetting"), gotoTut(1))
        # setting
        docContextTut += "setting.png", (lu("buttonStartSimulation"), handle._materialEditor.onStartSimulation)
        # wave propagation
        docContextTut += "wavePropagation.png", (lu("buttonMathModelsForWaves"), gotoMath(0)), (lu("buttonNumMethodsForWaveProp"), gotoMath(1)), None, \
                                                (lu("buttonChooseANewSetting"), handle.switchToMaterialEditor)
        # seismic imaging
        docContextTut += "seismicImaging1.png", (lu("buttonSeismicImagingLimitations"), gotoTut(4)), None, (lu("buttonMathOfSeismicInv"), gotoMath(2)), (lu("buttonNumFullWaveformInv"), gotoMath(3)), None, (lu("buttonChooseANewSetting"), handle.switchToMaterialEditor)
        docContextTut += "seismicImagingLimitations.png", (lu("buttonNumFullWaveformInv"), gotoMath(3)), (lu("buttonBackToSeismicImaging"), gotoTut(3))
        docContextTut += "seismicImaging2.png", (lu("buttonMathOfSeismicInv"), gotoMath(2)), (lu("buttonNumFullWaveformInv"), gotoMath(3))
        docContextTut += "credits.png"
        # open first page
        self._docWidgetTut.goTo(0, exception=False)

        # supply math context
        # mathematical models for waves
        docContextMath += "mathematicalModels.png", (lu("buttonNumMethodsForWaveProp"), gotoMath(1)), None, (lu("buttonBackToTutorial"), gotoTut())
        # numerical methods for wave propagation
        docContextMath += "numericalMethods.png", (lu("buttonMathModelsForWaves"), gotoMath(0)), None, (lu("buttonBackToTutorial"), gotoTut())
        # the mathematics of seismic inversion
        docContextMath += "mathematicsOfSeismicInversion.png", (lu("buttonMathOfSeismicInv"), gotoMath(3)), (lu("buttonNumMethodsForWaveProp"), gotoMath(1)), None, (lu("buttonBackToTutorial"), gotoTut())
        # numerical full waveform inversion
        docContextMath += "numericalFullWaveformInversion.png", (lu("buttonNumFullWaveformInv"), gotoMath(3)), (lu("buttonNumMethodsForWaveProp"), gotoMath(1)), None, (lu("buttonBackToTutorial"), gotoTut())
        # open first page
        self._docWidgetMath.goTo(0, exception=False)

        # add both documentation widgets to tab
        self.addTab(self._docWidgetTut, lu("tabTutorial"))
        self.addTab(self._docWidgetMath, lu("tabMath"))
        
    def paintEvent(self, paintEvent):        
        super(DocumentationTabWidget,self).paintEvent(paintEvent)
        qpainter = QPainter(self)   
        headerInfo = self.currentWidget().getCurrentPageString()        
        rect = qpainter.fontMetrics().boundingRect(headerInfo).adjusted(-1, 0, 0, 0)
        rect.moveRight(self.width()-5)
        rect.moveTop(0)
        qpainter.drawText(rect, Qt.AlignCenter, headerInfo)


class _DocPage():
    def __init__(self, fileName, *actionTuples):
        self._fileName = fileName
        self._name = fileName[:fileName.rfind(".")]
        self._actionTuples = actionTuples
        # image
        self._image = QImage(luDir(ResPath.documentation) + self._fileName)
    
    def getImage(self):
        return self._image
    
    def getActionWidget(self):
        # need to create widget here, because Qt deletes it when page is changed
        if self._actionTuples:
            widget = QWidget()
            layout = QVBoxLayout(widget)
            for tup in self._actionTuples:
                if tup is None:
                    layout.addSpacing(15)
                else:
                    name, func = tup
                    layout.addWidget(QPushButton(name, clicked=func))
            return widget
        else:
            return None

    def __eq__(self, name):
        return self._name == name

class DocumentationContext():
    def __init__(self):
        self._pages = []

    def getPage(self, index):
        return self._pages[index]

    def index(self, name):
        return self._pages.index(name)

    def addPage(self, fileName, *actionTuples):
        """
        appends a new page to this context
        @param fileName: the name of the file to load the image
        @param actionTuples: tuples of (name, function) pairs
        """
        self._pages.append(_DocPage(fileName, *actionTuples))

    def __len__(self):
        return len(self._pages)

    def __getitem__(self, index):
        return self._pages[index]

    def __iadd__(self, pageInfo):
        if isinstance(pageInfo, tuple):
            self.addPage(pageInfo[0], *pageInfo[1:])
        else:
            self.addPage(pageInfo)
        return self

    def __iter__(self):
        return iter(self._pages)


class DocumentationWidget(QWidget):
    def __init__(self, docContext, parent=None):
        QWidget.__init__(self, parent)
        
        #remember the original parent:
        self._originalParent = parent
        
        # the documentation context
        self._docContext = docContext

        self._currentIndex = 0
        # layout
        mainLayout = QGridLayout(self)
        # zoom button
#        buttonZoom = QPushButton(QIcon(ResPath.icons + "zoomIn.png"), lu("buttonMaximize"), clicked=self.zoom)
#        buttonZoom.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
#        buttonZoom.setMaximumWidth(buttonZoom.sizeHint().width())
#        mainLayout.addWidget(buttonZoom, 0, 0)

        # widget to show the current information image
        self._scrollArea = ScrollArea()
        mainLayout.addWidget(self._scrollArea, 1, 0, 1, 3)
        # navigation buttons
        buttonHome = QPushButton(lu("buttonHome"), clicked=self.home)
        buttonHome.setIcon(QIcon(ResPath.icons + "home.png"))
        mainLayout.addWidget(buttonHome, 2, 0)
        self._buttonPrev = QPushButton(lu("buttonPrevious"), enabled=False, clicked=self.previous)
        mainLayout.addWidget(self._buttonPrev, 2, 1)
        self._buttonNext = QPushButton(lu("buttonNext"), enabled=False, clicked=self.next)
        mainLayout.addWidget(self._buttonNext, 2, 2)
        
        # switch to first documentation
        self.goTo(0, exception=False)
        
        #init headerinfo
        self.updateHeaderInfo()
        
    def updateDocumentationContext(self, docContext):
        self._docContext = docContext

    def updateHeaderInfo(self):
        self.headerInfo = lu("labelPage") + " {}/{}".format(self._currentIndex + 1, len(self._docContext))
        #self._header.setText(text)

    def updateButtons(self):
        self._buttonPrev.setEnabled(self._currentIndex > 0)
        self._buttonNext.setEnabled(self._currentIndex < len(self._docContext) - 1)

    def goTo(self, index, exception=True):
        if 0 <= index < len(self._docContext):
            self._currentIndex = index
            page = self._docContext[index]

            self._scrollArea.setContent(page.getImage(), page.getActionWidget())
            self._scrollArea.verticalScrollBar().setValue(0)
        elif exception:
            raise IndexError()

        self.updateHeaderInfo()
        self.updateButtons()        
        self._originalParent.update()
        
    def previous(self):
        if self._currentIndex > 0:
            self.goTo(self._currentIndex - 1)

    def next(self):
        if self._currentIndex < len(self._docContext) - 1:
            self.goTo(self._currentIndex + 1)

    def home(self):
        self.goTo(0, exception=False)

    appTitle = lu("title")
    def zoom(self):
        docPage = self._docContext[self._currentIndex]
        d = ImageViewerDialog(docPage.getImage(), parent=self, title=DocumentationWidget.appTitle)
        d.show()
        d.containInWindow()
        d.exec_()
        
    def getCurrentPageString(self):
        return self.headerInfo

class ScrollArea(QScrollArea):
    def __init__(self, parent=None):
        QScrollArea.__init__(self, parent)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setBackgroundRole(QPalette.Light)

        self._container = None
        self.setWidget(self._container)

    def setContent(self, image, actionWidget):
        self._container = _Container(self, image, actionWidget)
        self.setWidget(self._container)
        self._resizeContainer()

    def resizeEvent(self, event):
        QScrollArea.resizeEvent(self, event)
        self._resizeContainer()

    def _resizeContainer(self):
        if not self._container:
            return

        width = self.width()
        height = self._container.getHeightForWidth(width)
        # if the image height is too big -> adjust width
        if height > self.height():
            width -= self.verticalScrollBar().width() + 1
            height = self._container.getHeightForWidth(width)
        self._container.resize(width, height)

class _Container(QWidget):
    def __init__(self, parent, image, actionWidget):
        QWidget.__init__(self, parent)
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        
        self._imageLabel = QLabel()
        self._imageLabel.setScaledContents(True)
        if image:
            self._imageLabel.setPixmap(QPixmap.fromImage(image))
        layout.addWidget(self._imageLabel)
        
        if actionWidget:
            layout.addWidget(actionWidget)

    def getHeightForWidth(self, width):
        if self._imageLabel.pixmap() and self._imageLabel.pixmap().width() > 0:
            ratio = self._imageLabel.pixmap().height() / float(self._imageLabel.pixmap().width())
            height = width * ratio
            return height + (self.height() - self._imageLabel.height())
        else:
            return self.height()
