import cStringIO

from PIL import Image
from PIL.ImageQt import ImageQt
from PyQt5.QtCore import QBuffer, QIODevice, QPoint, QRect
from PyQt5.QtGui import QImage, QPainter
import matplotlib

from lib.constants import Orientation
import numpy as np
from lib import constants


def qImageToPillow(qtImage):
    """
    converts a QImage to a PIL-Image.
    @param qtImage: the QImage
    @return: a Pillow image
    """
    buf = QBuffer()
    buf.open(QIODevice.ReadWrite)
    qtImage.save(buf, "PNG")
    strIO = cStringIO.StringIO()
    strIO.write(buf.data())
    buf.close()
    strIO.seek(0)
    return Image.open(strIO)

def qImageToArray(img, matColor):
    """
    converts a QImage object to a numpy array.
    @param img: the QImage to convert
    @param matColor: the color for the material
    @return: a numpy-array representing the QImage, where all entries with color matColor have the value constants.MATERTIAL_VALUE, else value 0
    """
    pilImage = qImageToPillow(img)
    
    arr = np.asarray(pilImage, dtype=np.float)

    # filter out all material pixels
    hardMask = arr == matColor
    hardMask = np.all(hardMask, axis=2)
    # multiply masks with suitable values and add them
    arr = hardMask * constants.MATERTIAL_VALUE
    
    return arr.T

_bgColor = np.array(constants.COLOR_BACKGROUND) / 255.0
_matColor = np.array(constants.COLOR_MATERIAL) / 255.0
_cmap_data = [(0.0, _bgColor), (1.0, _matColor)]
_cmap = matplotlib.colors.LinearSegmentedColormap.from_list('arrToMaterial', _cmap_data, N=256)
_cmap_amp_data = [(0.0, _bgColor), (1.0, _matColor / constants.CMAP_AMP_FACTOR)]
_cmap_amp = matplotlib.colors.LinearSegmentedColormap.from_list('arrToMaterialAmp', _cmap_amp_data, N=256)

def arrayToQImage(arr, amplify=False):
    """
    converts a numpy array to a QImage.
    @param arr: the numpy array to convert
    @return: a QImage representing the array. Zeros will have the constants.COLOR_BACKGROUND and values of constants.MATERTIAL_VALUE will get converted to constants.COLOR_MATERIAL
    """    
    cmap = _cmap_amp if amplify else _cmap 
    sm = matplotlib.cm.ScalarMappable(cmap=cmap)
    sm.set_clim(0.0, constants.MATERTIAL_VALUE)
    res = sm.to_rgba(arr.T, bytes=True)
    pilImage = Image.fromarray(res)
    return ImageQt(pilImage)

def createFilledImage(srcImage, targetSize, tilesX=1, tilesY=1):
    """
    creates a image of specified size that is filled with a given image.
    @param srcImage: the QImage to repeat
    @param targetSize: the desired image (-output) size as QSize
    @param tilesX: into how many parts should the targetRect be divided horizontally,
                    any value <= 0 means that this adjusts the srcImage width by as little as possible to repeat it inside the targetRect
    @param tilesY: into how many parts should the targetRect be divided vertically,
                    any value <= 0 means that this adjusts the srcImage height by as little as possible to repeat it inside the targetRect
    @return: the filled QImage
    """
    if tilesX <= 0:
        tilesX = max(1, targetSize.width() // srcImage.width())
    if tilesY <= 0:
        tilesY = max(1, targetSize.height() // srcImage.height())

    tileWidth = targetSize.width() // tilesX
    tileHeight = targetSize.height() // tilesY
    if tileWidth * tilesX != targetSize.width() or tileHeight * tilesY != targetSize.height():
        print "Warning, targetSize {} is no multiple of tileSize {}".format((tileWidth, tileHeight), (targetSize.width(), targetSize.height()))
    image = srcImage.scaled(tileWidth, tileHeight)  # , transformMode=Qt.SmoothTransformation)

    outputImage = QImage(targetSize, srcImage.format())
    outputImage.fill(0)
    painter = QPainter(outputImage)

    for iy in range(tilesY):
        for ix in range(tilesX):
            painter.drawImage(QPoint(ix * tileWidth, iy * tileHeight), image)

    return outputImage

def mergeImages(images, orientation=Orientation.VERTICAL):
    """
    clips images together along the specified orientation.
    @param images: an iterable of QImages
    @keyword orientation: merge orientation, either Orientation.VERTICAL or Orientation.HORIZONTAL. Default is Orientation.VERTICAL.
    @return: the output QImage
    the width or height (depends on the orientation) of the output image corresponds to the width or height of the first image.
    """
    width = images[0].width()
    height = images[0].height()

    for image in images[1:]:
        if orientation == Orientation.HORIZONTAL:
            width += image.width()
        elif orientation == Orientation.VERTICAL:
            height += image.height()

    outputImage = QImage(width, height, images[0].format())
    outputImage.fill(0)
    painter = QPainter(outputImage)
    x = 0
    y = 0

    for image in images:
        if orientation == Orientation.HORIZONTAL:
            painter.drawImage(QRect(x, y, image.width(), height), image)
            x += image.width()
        elif orientation == Orientation.VERTICAL:
            painter.drawImage(QRect(x, y, width, image.height()), image)
            y += image.height()
            
    painter.end()

    return outputImage

def qImageToHash(qImage):
    pilImage = qImageToPillow(qImage)
    arr = np.asarray(pilImage, dtype=np.float)
    return hash(bytes(arr.data))
    
#     ptr = qImage.bits()
#     ptr.setsize(qImage.byteCount())
#     return hash(ptr)
