from __future__ import print_function

import time
import os
import signal
from multiprocessing import Process, Event
from multiprocessing.queues import Queue, Empty

import threading
import random

import thread # python2 only

class ResetException(Exception):
    pass

class MyQueue(Queue):
    def clear(self):
        try:
            while True:
                a = self.get_nowait()
                print("Discarding '{}'".format(a))
        except Empty:
            pass


class MyProcess(Process):
    def signal_handler_INT(self, signum, frame):
        print(os.getpid(), "Ignoring signal", signum)
    
    def signal_handler_FPE(self, signum, frame):
        print(os.getpid(), "Main process wants us to reset")
        raise ResetException
    
    
    def __init__(self, q):
        self.q = q
        self.reset_counter = 0
        Process.__init__(self)
    
    
    def hard_work(self):
        while True:
            time.sleep(0.2)
            self.q.put("Run {}".format(self.reset_counter))
        
    def run(self):
        signal.signal(signal.SIGINT, self.signal_handler_INT) # prevent STRG+C from killing child
        signal.signal(signal.SIGFPE, self.signal_handler_FPE) # abuse SIGFPE for communication
        while True:
            try:
                self.hard_work()
            except ResetException:
                print(os.getpid(), "Stopped by event - resetting...")
                self.reset_counter += 1


if __name__ == "__main__":
    print("Main program")
    
    q = MyQueue()
    
    p = MyProcess(q)
    p.start()
    
    for i in range(3):
        try:
            print()
            t = 0.2 + 2*random.random()
            print("Main process waiting for {:.3f} seconds".format(t))
            time.sleep(t)
            print("Main process waiting done: resetting child")
            os.kill(p.pid, signal.SIGFPE)
            
            print("Example result:", q.get())
            
            q.clear()
            
        except KeyboardInterrupt:
            print("Stopping main program")
            break
        
    p.terminate()
    p.join()
    print("Finished")
