from PyQt5.QtCore import QEvent, QTimer, Qt, QSize, QRectF, QPointF
from PyQt5.QtGui import QIcon, QImage
from PyQt5.QtWidgets import QApplication, QHBoxLayout, QMainWindow, QTabWidget, QWidget

import instance
from lib import constants
from lib.constants import ResPath
from lib.i18n import lu, luDir
from widgets.creditstab import CreditsLabel
from widgets.inversion.inversiontab import InversionWidget
from widgets.materialeditor.materialeditortab import MaterialEditorWidget
from widgets.overlaywidget import OverlayWidget
from widgets.simulation.simulationtab import AnimationWidget

if not instance.args().beamer:
    from widgets.documentation import DocumentationTabWidget


class Window(QMainWindow):
    labelClickHint = lu("labelClickHint")

    def __init__(self):
        QMainWindow.__init__(self)

        self.introductionImage = QImage(luDir(ResPath.documentation) + "overview_overlay.png")
        self.speechBubbleImage = QImage(ResPath.textures + "speechBubble.png")
        
        self.setWindowTitle(lu("title"))
        self.setWindowIcon(QIcon(constants.ResPath.icons + "logo.png"))
        # set default size
        self.resize(QSize(constants.DEFAULT_WIDTH, constants.DEFAULT_HEIGHT))

        presentationMode = instance.args().beamer
        if not presentationMode:
            dummy = QWidget()
            mainLayout = QHBoxLayout(dummy)
            self.setCentralWidget(dummy)

        #=======================================================================
        # on left side is the tab widget for all interactions
        #=======================================================================
        self._interactionTab = QTabWidget()
        if presentationMode:
            self.setCentralWidget(self._interactionTab)
        else:
            mainLayout.addWidget(self._interactionTab, stretch=2)
        # material editor widget
        self._materialEditor = MaterialEditorWidget(self)
        self._interactionTab.addTab(self._materialEditor, lu("tabMaterialProperties"))
        # animation widget
        self._animWidget = AnimationWidget(self)
        self._interactionTab.addTab(self._animWidget, lu("tabAnimation"))
        self._interactionTab.setTabEnabled(1, False)
        # inversion tab
        self._inversionWidget = InversionWidget(self)
        self._interactionTab.addTab(self._inversionWidget, lu("tabInversion"))
        self._interactionTab.setTabEnabled(2, False)
        # credits tab
        self._interactionTab.addTab(CreditsLabel(), lu("tabCredits"))
        # test widget
        #self._interactionTab.addTab(TestWidget(self), "Test")

        #=======================================================================
        # on the right side is the tab widget for all documentation (tut, math)
        #=======================================================================
        if not presentationMode:
            self._documentationTab = DocumentationTabWidget(self)
            mainLayout.addWidget(self._documentationTab, stretch=1)
       

        #=======================================================================
        # Overlay
        #=======================================================================
        self._overlay = OverlayWidget(self.centralWidget())
        self._overlay.setHidden(True)

        self.forceTabUpdate(self._interactionTab)

    def postInit(self):
        if not instance.args().beamer:
            self.showIntroductionOverlay()

    def forceTabUpdate(self, tabWidget):
        self.setAttribute(Qt.WA_DontShowOnScreen)
        self.show()

        for i in range(1, tabWidget.count()):
            tabWidget.setCurrentIndex(i)

        self.hide()
        self.setAttribute(Qt.WA_DontShowOnScreen, on=False)
        tabWidget.setCurrentIndex(0)

    def switchToSimulation(self, material, emitterPos, receiverPos, centerXEmitRec):
        self._interactionTab.setTabEnabled(2, False)

        self._animWidget.reset()
        self._animWidget.loadFromData(material, emitterPos, receiverPos, centerXEmitRec)
        self._animWidget.start()
    
        # switch tab
        self._interactionTab.setTabEnabled(1, True)
        self._interactionTab.setCurrentIndex(1)
        
        if not instance.args().beamer:
            # switch documentation page TODO maybe don't hard-code this
            self._documentationTab.setCurrentIndex(0)
            self._documentationTab._docWidgetTut.goTo(2)

    def switchToInversion(self, realMaterial, seismoImage, centerXEmitRec):
        self._animWidget.pause()
        
        self._inversionWidget.reset()
        self._inversionWidget.loadFromData(realMaterial, seismoImage, centerXEmitRec)
        # switch tab
        self._interactionTab.setTabEnabled(2, True)
        self._interactionTab.setCurrentIndex(2)

        if not instance.args().beamer:
            # switch documentation page TODO maybe don't hard-code this
            self._documentationTab.setCurrentIndex(0)
            self._documentationTab._docWidgetTut.goTo(3)

    def switchToMaterialEditor(self):
        self._interactionTab.setCurrentIndex(0)
        if not instance.args().beamer:
            self._documentationTab._docWidgetTut.goTo(1)

    def getOverlay(self):
        return self._overlay

    def resizeEvent(self, event):
        QMainWindow.resizeEvent(self, event)        
        self._overlay.resize(event.size())
        event.accept()

    def reset(self):
        print "reset"
        self._interactionTab.setTabEnabled(1, False)
        self._interactionTab.setTabEnabled(2, False)
        self.switchToMaterialEditor()
        
        for i in range(self._interactionTab.count()):
            tabWidget = self._interactionTab.widget(i)
            tabWidget.reset()

        if not instance.args().beamer:
            self._documentationTab.setCurrentIndex(0)
            for i in range(self._documentationTab.count()):
                tabWidget = self._documentationTab.widget(i)
                tabWidget.home()

            self.showIntroductionOverlay()

    def showIntroductionOverlay(self):
        self._overlay.hide(animate=False)

        # add intro image
        self._overlay.addImage(QRectF(0.01, 0, 0.78, 1), self.introductionImage)
        # add speech bubble image
        bubbleRect = QRectF(0.8, 0.01, 0.2, 0.4)
        self._overlay.addImage(bubbleRect, self.speechBubbleImage)
        # add clicking hint
        text = Window.labelClickHint
        self._overlay.addText(bubbleRect, text, textSize=18, color=(0, 0, 0))

        self._overlay.show(animate=False, color=(150, 150, 150, 240))



class Application(QApplication):
    _resetEvents = (QEvent.KeyPress, QEvent.KeyRelease, QEvent.MouseButtonPress, QEvent.MouseButtonRelease, QEvent.MouseMove, QEvent.Wheel)

    def __init__(self, *args, **kwargs):
        QApplication.__init__(self, *args, **kwargs)

        self._idleWarningText = lu("idleWarning")

        self._idleWarningTimer = None
        self._idleOverlayTimer = None
        self._ignoreFlag = False

    def postInit(self, window):
        self._window = window
        window.postInit()

        self._idleWarningTimer = QTimer(self, timeout=self.resetWarning)
        self._idleWarningTimer.setSingleShot(True)
        self._idleWarningTimer.setInterval(constants.MAX_IDLE_TIME)

        self._idleOverlayTimer = QTimer(self, timeout=self.reset)
        self._idleOverlayTimer.setSingleShot(True)
        self._idleOverlayTimer.setInterval(constants.IDLE_WARNING_TIME)

        self._idleWarningTimer.start()

    def notify(self, receiver, event):
        if not self._ignoreFlag and event.type() in Application._resetEvents:
            self.resetTimer()
        return QApplication.notify(self, receiver, event)

    def resetTimer(self):
        if self._idleWarningTimer:
            if self._idleOverlayTimer and self._idleOverlayTimer.isActive():
                self._idleOverlayTimer.stop()
                self._window.getOverlay().hide()
            self._idleWarningTimer.start()

    def resetWarning(self):
        self._ignoreFlag = True
        self._window.getOverlay().hide(animate=False)
        self._window.getOverlay().addText((QPointF(0.2, 0.49)), self._idleWarningText)
        self._window.getOverlay().show()
        self.processEvents()
        self._ignoreFlag = False
        self._idleOverlayTimer.start()

    def reset(self):
        self._window.reset()
        # (re-) start idle timer
        self.resetTimer()

    def getAspectRatio(self):
        geom = self.desktop().availableGeometry()
        return geom.width() / float(geom.height())
